import org.codehaus.groovy.grails.plugins.springsecurity.Secured

@Secured(['ROLE_Manager','ROLE_AppAdmin'])
class PersonController extends BaseAppAdminController {

    def authenticateService
    def filterService

    // the delete, save and update actions only accept POST requests
    static Map allowedMethods = [delete: 'POST', save: 'POST', update: 'POST']

    def index = {
        redirect action: list, params: params
    }

    def list = {
        params.max = Math.min( params.max ? params.max.toInteger() : 10,  100 )

        if(!params.filter)
        { return [personList: Person.list(params), personTotal: Person.count()] }

        // filterPane:
        return[ personList: filterService.filter( params, Person ),
            personTotal: filterService.count( params, Person ),
            filterParams: com.zeddware.grails.plugins.filterpane.FilterUtils.extractFilterParams(params),
            params:params ]
    }

    def show = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Show)
        { params.action='show' }

        def person = Person.get(params.id)
        if (!person) {
            flash.message = "Person not found with id $params.id"
            redirect action: list
            return
        }
        List roleNames = []
        for (role in person.authorities) {
            roleNames << role.authority
        }
        roleNames.sort { n1, n2 ->
            n1 <=> n2
        }
        [person: person, roleNames: roleNames]
    }

    /**
    * Person delete action. Before removing an existing person,
    * they should be removed from those authorities which they are involved.
    */
    def delete = {

        def person = Person.get(params.id)
        if (person) {
            def authPrincipal = authenticateService.principal()
            // Avoid self-delete.
            if (!(authPrincipal instanceof String) && authPrincipal.username == person.loginName) {
                flash.message = "You cannot delete yourself, please login as another manager and try again."
                redirect(action:show,id:params.id)
            }
            else {
                //first, delete this person from Persons_Authorities table.
                Authority.findAll().each { it.removeFromPersons(person) }
                person.isActive = false
                person.save(flush: true)

                try {
                    person.delete(flush: true)
                    flash.message = "Person $params.id deleted."
                    redirect(action:list)
                }
                catch(org.springframework.dao.DataIntegrityViolationException e) {
                    flash.message = "Could not delete '$person.loginName' due to database constraints, but all authorities have been removed."
                    redirect(action:show,id:params.id)
                }
            }
        }
        else {
            flash.message = "Person not found with id $params.id"
        }
    }

    def edit = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Edit)
        { params.action='edit' }

        def person = Person.get(params.id)
        if (!person) {
            flash.message = "Person not found with id $params.id"
            redirect action: list
            return
        }
        params.message = "To allow login at least the 'ROLE_AppUser' authority must be given."
        return buildPersonModel(person)
    }

    /**
        * Person update action.
        */
    def update = {

        def person = Person.get(params.id)
        if (!person) {
            flash.message = "Person not found with id $params.id"
            redirect action: edit, id: params.id
            return
        }

        long version = params.version.toLong()
        if (person.version > version) {
            person.errors.rejectValue 'version', "person.optimistic.locking.failure",
                "Another user has updated this Person while you were editing."
            render view: 'edit', model: buildPersonModel(person)
            return
        }

        person.properties = params

        if(params.pass == "") {
            person.pass = "InsertNothingToClearValidation"
        }
        else {
            if (person.validate()) {
                person.password = authenticateService.encodePassword(params.pass)
            }
        }

        if (!person.hasErrors() && person.save(flush: true)) {
            Authority.findAll().each { it.removeFromPersons(person) }
            addRoles(person)
            flash.message = "Person '$params.id - $params.loginName' updated."
            redirect action: show, id: person.id
        }
        else {
            render view: 'edit', model: buildPersonModel(person)
        }

    }

    def create = {
        params.message = "To allow login at least the 'ROLE_AppUser' authority must be given."
        [person: new Person(params), authorityList: Authority.list()]
    }

    /**
        * Person save action.
        */
    def save = {

        def person = new Person()
        person.properties = params
        person.password = authenticateService.encodePassword(params.pass)
        if (person.save(flush: true)) {
            addRoles(person)
            redirect action: show, id: person.id
        }
        else {
            render view: 'create', model: [authorityList: Authority.list(), person: person]
        }
    }

    private void addRoles(person) {
        for (String key in params.keySet()) {
            if (key.contains('ROLE') && 'on' == params.get(key)) {
                Authority.findByAuthority(key).addToPersons(person)
            }
        }
    }

    private Map buildPersonModel(person) {

        List roles = Authority.list()
        roles.sort { r1, r2 ->
            r1.authority <=> r2.authority
        }
        Set userRoleNames = []
        for (role in person.authorities) {
            userRoleNames << role.authority
        }
        LinkedHashMap<Authority, Boolean> roleMap = [:]
        for (role in roles) {
            roleMap[(role)] = userRoleNames.contains(role.authority)
        }

        return [person: person, roleMap: roleMap]
    }
}
