/**
* Service class that encapsulates the business logic for InventoryItem searches.
*/
class InventoryItemSearchService {

    boolean transactional = false

    def messageSource

    def paramsMax = 100000

    /**
    * Selects and returns the correct search results based on the supplied quickSearch.
    * @param params The request params, may contain params.quickSearch string to specify the search.
    * @param locale The locale to use when generating result.message.
    */
    def getQuickSearch(params, locale) {
        def result = [:]
        result.quickSearch = params.quickSearch ?: "all"

        def getMessage = { Map m ->
            messageSource.getMessage(m.code, m.args == null ? null : m.args.toArray(), locale)
        }

        switch (result.quickSearch) {
            case "inventoryBelowReorder":
                result.inventoryItemList = getInventoryBelowReorder(params)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.description")
                else
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.none.found")
                break
            case "inventoryBelowReorderAll":
                result.inventoryItemList = getInventoryBelowReorder(params, false)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.all.description")
                else
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.none.found")
                break
            default:
                result.inventoryItemList = getAll(params)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.all.description")
                else
                    result.message = getMessage(code:"inventoryItem.search.text.all.none.found")
                break
        } // switch.

        // Success.
        return result

    } // getQuickSearch

    /**
    * Get all inventory items.
    * @param params The request params.
    */
    def getAll(params) {
        params.max = Math.min(params?.max?.toInteger() ?: 10, paramsMax)
        params.offset = params?.offset?.toInteger() ?: 0
        params.sort = params?.sort ?: "name"
        params.order = params?.order ?: "asc"

        def inventoryItemList = InventoryItem.createCriteria().list(
            max: params.max,
            offset: params.offset,
            sort: params.sort,
            order: params.order) {
            } // createCriteria
    } // getAll

    /**
    * List inventory items that are below reorder point.
    * @param params The request params.
    * @param onlyReorderEnabled Only include items with reorder enabled, defaults to true.
    */
    def getInventoryBelowReorder(params, onlyReorderEnabled=true) {
        params.max = Math.min(params?.max?.toInteger() ?: 10, paramsMax)
        params.offset = params?.offset?.toInteger() ?: 0
        params.sort = params?.sort ?: "name"
        params.order = params?.order ?: "asc"

        def inventoryItemList = InventoryItem.createCriteria().list(
            max: params.max,
            offset: params.offset,
            sort: params.sort,
            order: params.order) {
                eq("isActive", true)
                if(onlyReorderEnabled)
                    eq("enableReorder", true)
                leProperty("unitsInStock", "reorderPoint")
            } // createCriteria
    } // getInventoryBelowReorder

} // end class