import org.codehaus.groovy.grails.plugins.springsecurity.Secured

@Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager'])
class InventoryMovementDetailedController extends BaseController {

    def inventoryMovementService

    // the delete, save and update actions only accept POST requests
    static allowedMethods = [delete:'POST', save:'POST']

    def index = { redirect(action:list,params:params) }

    def list = {
        flash.message = flash.message
        redirect(controller:"taskDetailed", action:"search")
    }

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def show = {
        def inventoryMovementInstance = InventoryMovement.get( params.id )

        if(!inventoryMovementInstance) {
            flash.message = "InventoryMovement not found with id ${params.id}"
            redirect(action:list)
        }
        else { return [ inventoryMovementInstance : inventoryMovementInstance ] }
    }

    /**
    * List the inventory movements for an InventoryItem.
    * @param params.inventoryItem.id The id of an existing inventory item.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def listInventoryMovements = {
        def inventoryItemInstance = InventoryItem.get(params.inventoryItem.id)

        if(!inventoryItemInstance) {
            flash.message = "Inventory item not found with id ${params.id}"
            redirect(controller:'inventoryItemDetailed', action: 'search')
        }

        params.max = Math.min( params.max ? params.max.toInteger() : 10,  100)
        def inventoryMovementList = InventoryMovement.findAllByInventoryItem(inventoryItemInstance, params)
        def inventoryMovementListTotal = InventoryMovement.countByInventoryItem(inventoryItemInstance)

        [ inventoryMovementList: inventoryMovementList,
            inventoryMovementListTotal:  inventoryMovementListTotal,
            inventoryItemInstance: inventoryItemInstance]
    }

    def delete = {
        def result = inventoryMovementService.reverseMove(params)

        if(!result.error) {
            flash.message = "InventoryMovement ${params.id} deleted"
            redirect(controller:"taskDetailed", action:"show", id:result.taskId)
        }
        else {
            if(result.inventoryMovementInstance) {
                render(view:'show',model:[inventoryMovementInstance:result.inventoryMovementInstance.attach()])
            }
            else {
                flash.message = "Could not delete inventory movement."
                redirect(controller:"taskDetailed", action:"search")
            }
        }
    }

    def create = {
        def inventoryMovementInstance = new InventoryMovement()
        inventoryMovementInstance.properties = params
        def inventoryMovementTypeList = InventoryMovementType.withCriteria { gt("id", 1L) } // Don't include "Used".

        return ['inventoryMovementInstance': inventoryMovementInstance,
                        inventoryMovementTypeList: inventoryMovementTypeList]
    }

    /**
    * Handles all inventory movements except those of type "Used".
     * The "Used" type is handled directly by InventoryItemDetailedController and useInventoryItem.
    */
    def save = {
        def result = inventoryMovementService.move(params)

        if(!result.error) {
            flash.message = "Inventory Movement for ${result.inventoryMovementInstance.inventoryItem.name} created."
            if(result.taskId) {
                redirect(controller:"taskDetailed", action:"show", id: result.taskId)
                return
            }
            redirect(controller:"inventoryItemDetailed", action:"show", id: result.inventoryMovementInstance.inventoryItem.id)
        }
        else {
            if(result.inventoryMovementInstance) {
                def inventoryMovementTypeList = InventoryMovementType.withCriteria { gt("id", 1L) } // Don't include "Used".
                render(view:'create', model:[inventoryMovementInstance: result.inventoryMovementInstance,
                                                            inventoryMovementTypeList: inventoryMovementTypeList])
            }
            else {
                flash.message = "Could not create inventory movement."
                redirect(controller:"taskDetailed", action:"search", id:result.taskId)
            }

        }
    }

}
